<?php
// Enable detailed error reporting
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Set JSON header
header('Content-Type: application/json');

// Basic security checks
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Only POST requests are allowed']));
}

// Required parameters check
$required = ['url', 'type', 'video_id'];
foreach ($required as $param) {
    if (!isset($_POST[$param])) {
        http_response_code(400);
        die(json_encode(['error' => "Missing parameter: $param"]));
    }
}

// Input sanitization
$videoUrl = filter_var($_POST['url'], FILTER_SANITIZE_URL);
$videoId = filter_var($_POST['video_id'], FILTER_SANITIZE_STRING);
$type = filter_var($_POST['type'], FILTER_SANITIZE_STRING);
$quality = $_POST['quality'] ?? 'best';
$audioQuality = $_POST['audio_quality'] ?? 'best';

// Validate YouTube URL and ID
if (!preg_match('/^(https?:\/\/)?(www\.)?(youtube\.com|youtu\.be)/', $videoUrl)) {
    http_response_code(400);
    die(json_encode(['error' => 'Invalid YouTube URL']));
}

if (!preg_match('/^[a-zA-Z0-9_-]{11}$/', $videoId)) {
    http_response_code(400);
    die(json_encode(['error' => 'Invalid YouTube video ID']));
}

// Create temporary directory
$tempDir = __DIR__ . '/temp_downloads';
if (!file_exists($tempDir)) {
    if (!mkdir($tempDir, 0755, true)) {
        http_response_code(500);
        die(json_encode(['error' => 'Could not create temp directory']));
    }
}

try {
    // Generate unique filename
    $filename = "yt_{$videoId}_" . time();
    $outputTemplate = "$tempDir/$filename.%(ext)s";

    // Build yt-dlp command based on type
    if ($type === 'audio') {
        // Audio download (MP3)
        $qualityFlag = $audioQuality === 'medium' ? '--audio-quality 5' : '--audio-quality 0';
        $command = "yt-dlp -x --audio-format mp3 $qualityFlag -o '$outputTemplate' --force-overwrites '$videoUrl' 2>&1";
    } else {
        // Video download
        $format = match($quality) {
            '1080' => 'bestvideo[height<=1080][ext=mp4]+bestaudio[ext=m4a]/best[ext=mp4][height<=1080]',
            '720' => 'bestvideo[height<=720][ext=mp4]+bestaudio[ext=m4a]/best[ext=mp4][height<=720]',
            default => 'bestvideo[ext=mp4]+bestaudio[ext=m4a]/best[ext=mp4]'
        };
        $command = "yt-dlp -f '$format' -o '$outputTemplate' --force-overwrites '$videoUrl' 2>&1";
    }

    // Execute command
    exec($command, $output, $returnCode);

    if ($returnCode !== 0) {
        throw new Exception("Download failed: " . implode("\n", $output));
    }

    // Find the downloaded file
    $files = glob("$tempDir/$filename.*");
    if (empty($files)) {
        throw new Exception("Downloaded file not found");
    }

    $downloadedFile = $files[0];
    $fileExt = pathinfo($downloadedFile, PATHINFO_EXTENSION);
    $finalFilename = "video_{$videoId}.$fileExt";
    $finalPath = "$tempDir/$finalFilename";

    // Rename to final filename
    if (!rename($downloadedFile, $finalPath)) {
        throw new Exception("Could not rename downloaded file");
    }

    // Return download information
    echo json_encode([
        'success' => true,
        'download_url' => "serve_file.php?file=" . urlencode($finalFilename),
        'filename' => $finalFilename,
        'type' => $type,
        'quality' => $type === 'audio' ? $audioQuality : $quality
    ]);

} catch (Exception $e) {
    // Clean up any partial files
    array_map('unlink', glob("$tempDir/$filename.*"));
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}